/*
 * Copyright 2019 Mozilla Foundation. See the COPYRIGHT
 * file at the top-level directory of this distribution.
 *
 * Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
 * https://www.apache.org/licenses/LICENSE-2.0> or the MIT license
 * <LICENSE-MIT or https://opensource.org/licenses/MIT>, at your
 * option. This file may not be copied, modified, or distributed
 * except according to those terms.
**/

/* clang-format off */


#ifndef mapped_hyph_h
#define mapped_hyph_h

/*
 * Warning, this file is autogenerated by cbindgen. Don't modify this manually.
 */


#include <stdbool.h>
#include <stdint.h>

/**
 * Opaque type representing a compiled dictionary in a memory buffer.
 */
typedef struct CompiledData CompiledData;

/**
 * Opaque type representing a hyphenation dictionary loaded from a file,
 * for use in FFI function signatures.
 */
typedef struct HyphDic HyphDic;

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/**
 * C-callable function to load a hyphenation dictionary from a file at `path`.
 *
 * Returns null on failure.
 *
 * This does not fully validate that the file contains usable hyphenation
 * data, it only opens the file (read-only) and mmap's it into memory, and
 * does some minimal sanity-checking that it *might* be valid.
 *
 * The returned `HyphDic` must be released with `mapped_hyph_free_dictionary`.
 *
 * # Safety
 * The given `path` must be a valid pointer to a NUL-terminated (C-style)
 * string.
 */
const struct HyphDic *mapped_hyph_load_dictionary(const char *path);

/**
 * C-callable function to free a hyphenation dictionary
 * that was loaded by `mapped_hyph_load_dictionary`.
 *
 * # Safety
 * The `dic` parameter must be a `HyphDic` pointer obtained from
 * `mapped_hyph_load_dictionary`, and not previously freed.
 */
void mapped_hyph_free_dictionary(struct HyphDic *dic);

/**
 * C-callable function to find hyphenation values for a given `word`,
 * using a dictionary loaded via `mapped_hyph_load_dictionary`.
 *
 * The `word` must be UTF-8-encoded, and is `word_len` bytes (not characters)
 * long.
 *
 * Caller must supply the `hyphens` output buffer for results; its size is
 * given in `hyphens_len`.
 * It should be at least `word_len` elements long.
 *
 * Returns -1 if `word` is not valid UTF-8, or the output `hyphens` buffer is
 * too small.
 * Otherwise returns the number of potential hyphenation positions found.
 *
 * # Panics
 * This function may panic if the given dictionary is not valid.
 *
 * # Safety
 * The `dic` parameter must be a `HyphDic` pointer obtained from
 * `mapped_hyph_load_dictionary`.
 *
 * The `word` and `hyphens` parameter must be valid pointers to memory buffers
 * of at least the respective sizes `word_len` and `hyphens_len`.
 */
int32_t mapped_hyph_find_hyphen_values_dic(const struct HyphDic *dic,
                                           const char *word,
                                           uint32_t word_len,
                                           uint8_t *hyphens,
                                           uint32_t hyphens_len);

/**
 * C-callable function to find hyphenation values for a given `word`,
 * using a dictionary loaded and owned by the caller.
 *
 * The dictionary is supplied as a raw memory buffer `dic_buf` of size
 * `dic_len`.
 *
 * The `word` must be UTF-8-encoded, and is `word_len` bytes (not characters)
 * long.
 *
 * Caller must supply the `hyphens` output buffer for results; its size is
 * given in `hyphens_len`.
 * It should be at least `word_len` elements long.
 *
 * Returns -1 if `word` is not valid UTF-8, or the output `hyphens` buffer is
 * too small.
 * Otherwise returns the number of potential hyphenation positions found.
 *
 * # Panics
 * This function may panic if the given dictionary is not valid.
 *
 * # Safety
 * The `dic_buf` parameter must be a valid pointer to a memory block of size
 * at least `dic_len`.
 *
 * The `word` and `hyphens` parameter must be valid pointers to memory buffers
 * of at least the respective sizes `word_len` and `hyphens_len`.
 */
int32_t mapped_hyph_find_hyphen_values_raw(const uint8_t *dic_buf,
                                           uint32_t dic_len,
                                           const char *word,
                                           uint32_t word_len,
                                           uint8_t *hyphens,
                                           uint32_t hyphens_len);

/**
 * C-callable function to check if a given memory buffer `dic_buf` of size
 * `dic_len` is potentially usable as a hyphenation dictionary.
 *
 * Returns `true` if the given memory buffer looks like it may be a valid
 * hyphenation dictionary, `false` if it is clearly not usable.
 *
 * # Safety
 * The `dic_buf` parameter must be a valid pointer to a memory block of size
 * at least `dic_len`.
 */
bool mapped_hyph_is_valid_hyphenator(const uint8_t *dic_buf, uint32_t dic_len);

/**
 * C-callable function to free a CompiledData object created by
 * a `mapped_hyph_compile_...` function (below).
 *
 * # Safety
 * The `data` parameter must be a `CompiledData` pointer obtained from
 * a `mapped_hyph_compile_...` function, and not previously freed.
 */
void mapped_hyph_free_compiled_data(struct CompiledData *data);

/**
 * C-callable function to compile hyphenation patterns from `pattern_buf` and return
 * the compiled data in a memory buffer, suitable to be stored somewhere or passed
 * to `mapped_hyph_find_hyphen_values_raw` to perform hyphenation.
 *
 * The returned `CompiledData` must be released with `mapped_hyph_free_compiled_data`.
 *
 * # Safety
 * The `pattern_buf` parameter must be a valid pointer to a memory block of size
 * at least `pattern_len`.
 */
const struct CompiledData *mapped_hyph_compile_buffer(const uint8_t *pattern_buf,
                                                      uint32_t pattern_len,
                                                      bool compress);

/**
 * C-callable function to compile hyphenation patterns from a file to a memory buffer.
 *
 * The returned `CompiledData` must be released with `mapped_hyph_free_compiled_data`.
 *
 * # Safety
 * The given `path` must be a valid pointer to a NUL-terminated (C-style) string.
 */
const struct CompiledData *mapped_hyph_compile_file(const char *path, bool compress);

/**
 * Get the size of the compiled table buffer in a `CompiledData` object.
 *
 * # Safety
 * The `data` parameter must be a `CompiledData` pointer obtained from
 * a `mapped_hyph_compile_...` function, and not previously freed.
 */
uint32_t mapped_hyph_compiled_data_size(const struct CompiledData *data);

/**
 * Get a pointer to the raw data held by a `CompiledData` object.
 *
 * # Safety
 * The `data` parameter must be a `CompiledData` pointer obtained from
 * a `mapped_hyph_compile_...` function, and not previously freed.
 *
 * The returned pointer only remains valid as long as the `CompiledData` has not
 * been released (by passing it to `mapped_hyph_free_compiled_data`).
 */
const uint8_t *mapped_hyph_compiled_data_ptr(const struct CompiledData *data);

#ifdef __cplusplus
}  // extern "C"
#endif  // __cplusplus

#endif  /* mapped_hyph_h */

/* clang-format on */
