// SPDX-FileCopyrightText: Copyright (c) 2008-2013, NVIDIA Corporation. All rights reserved.
// SPDX-License-Identifier: Apache-2.0

#pragma once

#include <thrust/detail/config.h>

#if defined(_CCCL_IMPLICIT_SYSTEM_HEADER_GCC)
#  pragma GCC system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_CLANG)
#  pragma clang system_header
#elif defined(_CCCL_IMPLICIT_SYSTEM_HEADER_MSVC)
#  pragma system_header
#endif // no system header
#include <thrust/detail/execution_policy.h>
#include <thrust/mr/allocator.h>

// Include all active backend system implementations (generic, sequential, host and device)
#include <thrust/system/detail/generic/per_device_resource.h>
#include <thrust/system/detail/sequential/per_device_resource.h>
#include __THRUST_HOST_SYSTEM_ALGORITH_DETAIL_HEADER_INCLUDE(per_device_resource.h)
#include __THRUST_DEVICE_SYSTEM_ALGORITH_DETAIL_HEADER_INCLUDE(per_device_resource.h)

// Some build systems need a hint to know which files we actually include
#if 0
#  include <thrust/system/cpp/detail/per_device_resource.h>
#  include <thrust/system/cuda/detail/per_device_resource.h>
#  include <thrust/system/omp/detail/per_device_resource.h>
#  include <thrust/system/tbb/detail/per_device_resource.h>
#endif

THRUST_NAMESPACE_BEGIN

/*! Returns a global instance of \p MR for the current device of the provided system.
 *
 *  \tparam MR type of a memory resource to get an instance from. Must be \p DefaultConstructible.
 *  \param system execution policy for which the resource is requested.
 *  \return a pointer to a global instance of \p MR for the current device.
 *
 *  \verbatim embed:rst:leading-asterisk
 *     .. versionadded:: 2.2.0
 *  \endverbatim
 */
template <typename MR, typename DerivedPolicy>
_CCCL_HOST MR* get_per_device_resource(const thrust::detail::execution_policy_base<DerivedPolicy>& system)
{
  using thrust::system::detail::generic::get_per_device_resource;

  return get_per_device_resource<MR>(thrust::detail::derived_cast(thrust::detail::strip_const(system)));
}

/*! A helper allocator class that uses global per device instances of a given upstream memory resource. Requires the
 * memory resource to be default constructible.
 *
 *  \tparam T the type that will be allocated by this allocator.
 *  \tparam MR the upstream memory resource to use for memory allocation. Must derive from
 *      \p thrust::mr::memory_resource and must be \p final.
 *  \tparam ExecutionPolicy the execution policy of the system to be used to retrieve the resource for the current
 * device.
 *
 *  \verbatim embed:rst:leading-asterisk
 *     .. versionadded:: 2.2.0
 *  \endverbatim
 */
template <typename T, typename Upstream, typename ExecutionPolicy>
class per_device_allocator : public thrust::mr::allocator<T, Upstream>
{
  using base = thrust::mr::allocator<T, Upstream>;

public:
  /*! The \p rebind metafunction provides the type of an \p per_device_allocator instantiated with another type.
   *
   *  \tparam U the other type to use for instantiation.
   *
   *  \verbatim embed:rst:leading-asterisk
   *     .. versionadded:: 2.2.0
   *  \endverbatim
   */
  template <typename U>
  struct rebind
  {
    /*! The alias \p other gives the type of the rebound \p per_device_allocator.
     */
    using other = per_device_allocator<U, Upstream, ExecutionPolicy>;
  };

  /*! Default constructor. Uses \p get_global_resource to get the global instance of \p Upstream and initializes the
   *      \p allocator base subobject with that resource.
   *
   *  \verbatim embed:rst:leading-asterisk
   *     .. versionadded:: 2.2.0
   *  \endverbatim
   */
  _CCCL_HOST per_device_allocator()
      : base(get_per_device_resource<Upstream>(ExecutionPolicy()))
  {}

  /*! Copy constructor. Copies the memory resource pointer. */
  _CCCL_HOST_DEVICE per_device_allocator(const per_device_allocator& other)
      : base(other)
  {}

  /*! Conversion constructor from an allocator of a different type. Copies the memory resource pointer. */
  template <typename U>
  _CCCL_HOST_DEVICE per_device_allocator(const per_device_allocator<U, Upstream, ExecutionPolicy>& other)
      : base(other)
  {}

  /*! Destructor. */
  _CCCL_HOST_DEVICE ~per_device_allocator() {}
};

THRUST_NAMESPACE_END
