/***************************************************************************
 *
 * tdenetman-connection_editor.cpp - A NetworkManager frontend for TDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa        <hschaa@suse.de>, <helmut.schaa@gmx.de>
 * Author: Timothy Pearson <kb9vqf@pearsoncomputing.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

// tqt headers
#include <tqwidget.h>
#include <tqcombobox.h>
#include <tqtabwidget.h>
#include <tqpushbutton.h>
#include <tqwidgetstack.h>
#include <tqapplication.h>
#include <tqlabel.h>
#include <tqpopupmenu.h>

// tde headers
#include <kiconloader.h>
#include <kdebug.h>
#include <kpushbutton.h>
#include <tdelistview.h>
#include <tdelocale.h>
#include <kcombobox.h>
#include <tdemessagebox.h>

// tdenm headers
#include "tdenetman-connection_editor.h"
#include "tdenetman-connection_settings_dialog.h"

extern unsigned int tdenetworkmanager_editor_dialog_count;

using namespace ConnectionSettings;

/*
 * ConnectionListViewItem
 */
class ConnectionListViewItem : public TDEListViewItem
{
	public:

	ConnectionListViewItem(TQListView* parent, TQString connection)
		: TDEListViewItem(parent)
		, _conn(connection)
	{
		TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
		TDENetworkConnection* conn = (nm)?nm->findConnectionByUUID(connection):NULL;

		if (conn) {
			setText(0, conn->friendlyName);
			setText(1, TDENetworkConnectionManager::friendlyConnectionTypeName(conn->type()));
			// TODO: Move to a Factory
			if (conn->type() == TDENetworkConnectionType::WiredEthernet) {
				setPixmap(0, TDEGlobal::iconLoader()->loadIcon("wired", TDEIcon::Small));
			}
			else if (conn->type() == TDENetworkConnectionType::WiFi) {
				setPixmap(0, TDEGlobal::iconLoader()->loadIcon("wireless", TDEIcon::Small));
			}
			else if (conn->type() == TDENetworkConnectionType::VPN) {
				setPixmap(0, TDEGlobal::iconLoader()->loadIcon("encrypted", TDEIcon::Small));
			}
			else {
				setPixmap(0, TDEGlobal::iconLoader()->loadIcon("help", TDEIcon::Small));
			}
		}
	}

	TQString _conn;
};

/*
 * Constructor
 */
ConnectionEditorImpl::ConnectionEditorImpl(TQWidget* parent, const char* name, bool modal, WFlags fl)
	: ConnectionEditor(parent, name, modal, fl)
{
	tdenetworkmanager_editor_dialog_count++;

	// TODO: enable combobox if implemented
	cboConnectionType->hide();

	// TODO: Editmode is not ready yet, hide the button
//	pbEdit->hide();


	pbNew->setIconSet(TDEGlobal::iconLoader()->loadIcon("add", TDEIcon::Small));
	pbDelete->setIconSet(TDEGlobal::iconLoader()->loadIcon("remove", TDEIcon::Small));
	pbEdit->setIconSet(TDEGlobal::iconLoader()->loadIcon("edit", TDEIcon::Small));

	TQPopupMenu* popup = new TQPopupMenu(pbNew);
	// TODO: move to a factory class
	popup->insertItem(TDEGlobal::iconLoader()->loadIcon("wireless", TDEIcon::Small), i18n("Wireless"), this, TQ_SLOT(slotNewWirelessConnection()));
	popup->insertItem(TDEGlobal::iconLoader()->loadIcon("wired", TDEIcon::Small), i18n("Wired"), this, TQ_SLOT(slotNewWiredConnection()));

// 	if (!VPNManager::getVPNServices().isEmpty()) {
		popup->insertItem(TDEGlobal::iconLoader()->loadIcon("encrypted", TDEIcon::Small), i18n("VPN"), this, TQ_SLOT(slotNewVPNConnection()));
// 	}

	pbNew->setPopup(popup);

	connect(pbClose, TQ_SIGNAL(clicked()), this, TQ_SLOT(close()));
	connect(pbDelete, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotRemoveCurrentConnection()));
	connect(pbEdit, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotEditCurrentConnection()));

	// show all connections
	fillConnectionList();
}

/*
 * Destructor
 */
ConnectionEditorImpl::~ConnectionEditorImpl()
{
	// remove the popupmenu
	if (pbNew->popup()) {
		delete pbNew->popup();
	}

	tdenetworkmanager_editor_dialog_count--;
}

/*
 * New Wireless connection
 */
void ConnectionEditorImpl::slotNewWirelessConnection()
{
	// create a new wireless connection
	slotEditNewConnection(new TDEWiFiConnection());
}

/*
 * New Wired connection
 */
void ConnectionEditorImpl::slotNewWiredConnection()
{
	slotEditNewConnection(new TDEWiredEthernetConnection());
}

/*
 * New VPN connection
 */
void ConnectionEditorImpl::slotNewVPNConnection()
{
	slotEditNewConnection(new TDEVPNConnection());
}

/*
 *
 */
void ConnectionEditorImpl::slotEditNewConnection(TDENetworkConnection* conn)
{
	// open a dialog for editing the connection
	ConnectionSettingsDialogImpl* dlg = new ConnectionSettingsDialogImpl(conn, true, TQByteArray(), this, "connect_something", false, TQt::WDestructiveClose);
	connect(dlg, TQ_SIGNAL(connectionSaved()), this, TQ_SLOT(slotRefreshConnectionList()));
	dlg->show();
}

void ConnectionEditorImpl::slotRefreshConnectionList()
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
	if (!nm) {
		return;
	}
	nm->loadConnectionInformation();
	fillConnectionList();
}

/*
 * Edit the selected connection
 */
void ConnectionEditorImpl::slotEditCurrentConnection()
{
	ConnectionListViewItem* item = dynamic_cast<ConnectionListViewItem*>(lvConnections->currentItem());
	if (!item) {
		return;
	}

	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
	if (!nm) {
		return;
	}

	TDENetworkConnection* conn = nm->findConnectionByUUID(item->_conn);

	if (conn) {
		// we need the secrets for editing
		nm->loadConnectionSecrets(conn->UUID);

		ConnectionSettingsDialogImpl* dlg = new ConnectionSettingsDialogImpl(conn, false, TQByteArray(), this, "connect_something", false, TQt::WDestructiveClose);
		dlg->show();
	}
}


/*
 * Delete the selected connection
 */
void ConnectionEditorImpl::slotRemoveCurrentConnection()
{
	ConnectionListViewItem* item = dynamic_cast<ConnectionListViewItem*>(lvConnections->currentItem());
	if (!item) {
		return;
	}

	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
	if (!nm) {
		return;
	}

	TDENetworkConnection* conn = nm->findConnectionByUUID(item->_conn);

	if (!nm->deleteConnection(conn->UUID)) {
		KMessageBox::error(this, i18n("<qt><b>Unable to delete network connection!</b><p>Potential causes:<br> * Insufficient permissions<br> * NetworkManager not running<br> * DBUS failure</qt>"), i18n("Unable to perform requested operation"));
	}

	slotRefreshConnectionList();
}

/*
 * Fill the connection list
 */
void ConnectionEditorImpl::fillConnectionList()
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
	if (!nm) {
		return;
	}

	lvConnections->clear();

	TDENetworkConnectionList* allconmap = nm->connections();
	for (TDENetworkConnectionList::Iterator it = allconmap->begin(); it != allconmap->end(); ++it) {
		TDENetworkConnection* conn = *it;
		new ConnectionListViewItem(lvConnections, conn->UUID);
	}
}

#include "tdenetman-connection_editor.moc"
