/**
 * defaulteditor.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
#include "defaulteditor.h"
#include "core.h"

#include <kgenericfactory.h>
#include <tdeapplication.h>
#include <tdeaction.h>
#include <kiconloader.h>
#include <kdebug.h>

#include <tdeaction.h>
#include <kcolordialog.h>
#include <tdefiledialog.h>
#include <tdeinstance.h>
#include <tdelocale.h>
#include <kstdaction.h>
#include <kprinter.h>
#include <kfinddialog.h>
#include <kfind.h>
#include <kreplacedialog.h>
#include <kreplace.h>

#include <tqtextedit.h>
#include <tqwidget.h>

typedef KGenericFactory<DefaultEditor> DefaultEditorFactory;
K_EXPORT_COMPONENT_FACTORY( libkomposer_defaulteditor,
                            DefaultEditorFactory( "komposer_defaulteditor" ) )

DefaultEditor::DefaultEditor( TQObject *parent, const char *name, const TQStringList &args )
  : Editor( parent, name, args ), m_textEdit( 0 )
{
  setInstance( DefaultEditorFactory::instance() );

  m_textEdit = new TQTextEdit( 0 );

  createActions( actionCollection() );

  setXMLFile( "defaulteditorui.rc" );
}

DefaultEditor::~DefaultEditor()
{
}


TQWidget*
DefaultEditor::widget()
{
    return m_textEdit;
}

TQString
DefaultEditor::text() const
{
  return m_textEdit->text();
}

void
DefaultEditor::setText( const TQString &text )
{
  m_textEdit->setText( text );
}

void
DefaultEditor::changeSignature( const TQString &sig )
{
  TQString text = m_textEdit->text();

  int sigStart = text.findRev( "-- " );
  TQString sigText = TQString( "-- \n%1" ).arg( sig );

  text.replace( sigStart, text.length(), sigText );
}

void
DefaultEditor::createActions( TDEActionCollection *ac )
{
  //
  // File Actions
  //
  (void) KStdAction::open( this, TQ_SLOT(open()), ac );
  (void) KStdAction::openRecent( this, TQ_SLOT(openURL(const KURL &)), ac );
  (void) KStdAction::save( this, TQ_SLOT(save()), ac );
  (void) KStdAction::saveAs( this, TQ_SLOT(saveAs()), ac );

  //
  // Edit Actions
  //
  TDEAction *actionUndo = KStdAction::undo( m_textEdit, TQ_SLOT(undo()), ac );
  actionUndo->setEnabled( false );
  connect( m_textEdit, TQ_SIGNAL(undoAvailable(bool)),
           actionUndo, TQ_SLOT(setEnabled(bool)) );

  TDEAction *actionRedo = KStdAction::redo( m_textEdit, TQ_SLOT(redo()), ac );
  actionRedo->setEnabled( false );
  connect( m_textEdit, TQ_SIGNAL(redoAvailable(bool)),
           actionRedo, TQ_SLOT(setEnabled(bool)) );

  TDEAction *action_cut = KStdAction::cut( m_textEdit, TQ_SLOT(cut()), ac );
  action_cut->setEnabled( false );
  connect( m_textEdit, TQ_SIGNAL(copyAvailable(bool)),
           action_cut, TQ_SLOT(setEnabled(bool)) );

  TDEAction *action_copy = KStdAction::copy( m_textEdit, TQ_SLOT(copy()), ac );
  action_copy->setEnabled( false );
  connect( m_textEdit, TQ_SIGNAL(copyAvailable(bool)),
           action_copy, TQ_SLOT(setEnabled(bool)) );

  (void) KStdAction::print( this, TQ_SLOT(print()), ac );

  (void) KStdAction::paste( m_textEdit, TQ_SLOT(paste()), ac );
  (void) new TDEAction( i18n( "C&lear" ), 0,
                      m_textEdit, TQ_SLOT(removeSelectedText()),
                      ac, "edit_clear" );

  (void) KStdAction::selectAll( m_textEdit, TQ_SLOT(selectAll()), ac );

  //
  // View Actions
  //
  (void) KStdAction::zoomIn( m_textEdit, TQ_SLOT(zoomIn()), ac );
  (void) KStdAction::zoomOut( m_textEdit, TQ_SLOT(zoomOut()), ac );

  //
  // Character Formatting
  //
  m_actionBold = new TDEToggleAction( i18n("&Bold"), "format-text-bold", CTRL+Key_B,
                                    ac, "format_bold" );
  connect( m_actionBold, TQ_SIGNAL(toggled(bool)),
           m_textEdit, TQ_SLOT(setBold(bool)) );

  m_actionItalic = new TDEToggleAction( i18n("&Italic"), "format-text-italic", CTRL+Key_I,
                                      ac, "format_italic" );

  connect( m_actionItalic, TQ_SIGNAL(toggled(bool)),
           m_textEdit, TQ_SLOT(setItalic(bool) ));

  m_actionUnderline = new TDEToggleAction( i18n("&Underline"), "format-text-underline", CTRL+Key_U,
                                         ac, "format_underline" );

  connect( m_actionUnderline, TQ_SIGNAL(toggled(bool)),
           m_textEdit, TQ_SLOT(setUnderline(bool)) );

  (void) new TDEAction( i18n("Text &Color..."), "colorpicker", 0,
                      this, TQ_SLOT(formatColor()),
                      ac, "format_color" );

  //
  // Font
  //
  m_actionFont = new TDEFontAction( i18n("&Font"), 0,
                                 ac, "format_font" );
  connect( m_actionFont, TQ_SIGNAL(activated(const TQString &)),
           m_textEdit, TQ_SLOT(setFamily(const TQString &)) );


  m_actionFontSize = new TDEFontSizeAction( i18n("Font &Size"), 0,
                                          ac, "format_font_size" );
  connect( m_actionFontSize, TQ_SIGNAL(fontSizeChanged(int)),
           m_textEdit, TQ_SLOT(setPointSize(int)) );

  //
  // Alignment
  //
  m_actionAlignLeft = new TDEToggleAction( i18n("Align &Left"), "format-text-direction-ltr", 0,
                                         ac, "format_align_left" );
  connect( m_actionAlignLeft, TQ_SIGNAL(toggled(bool)),
           this, TQ_SLOT(setAlignLeft(bool)) );

  m_actionAlignCenter = new TDEToggleAction( i18n("Align &Center"), "text_center", 0,
                                           ac, "format_align_center" );
  connect( m_actionAlignCenter, TQ_SIGNAL(toggled(bool)),
           this, TQ_SLOT(setAlignCenter(bool)) );

  m_actionAlignRight = new TDEToggleAction( i18n("Align &Right"), "format-text-direction-rtl", 0,
                                          ac, "format_align_right" );
  connect( m_actionAlignRight, TQ_SIGNAL(toggled(bool)),
           this, TQ_SLOT(setAlignRight(bool)) );

  m_actionAlignJustify = new TDEToggleAction( i18n("&Justify"), "text_block", 0,
                                            ac, "format_align_justify" );
  connect( m_actionAlignJustify, TQ_SIGNAL(toggled(bool)),
           this, TQ_SLOT(setAlignJustify(bool)) );

  m_actionAlignLeft->setExclusiveGroup( "alignment" );
  m_actionAlignCenter->setExclusiveGroup( "alignment" );
  m_actionAlignRight->setExclusiveGroup( "alignment" );
  m_actionAlignJustify->setExclusiveGroup( "alignment" );

  //
  // Tools
  //
  (void) KStdAction::spelling( this, TQ_SLOT(checkSpelling()), ac );

  //
  // Setup enable/disable
  //
  updateActions();

  connect( m_textEdit, TQ_SIGNAL(currentFontChanged(const TQFont &)),
           this, TQ_SLOT( updateFont() ) );
  connect( m_textEdit, TQ_SIGNAL(currentFontChanged(const TQFont &)),
           this, TQ_SLOT(updateCharFmt()) );
  connect( m_textEdit, TQ_SIGNAL(cursorPositionChanged(int, int)),
           this, TQ_SLOT(updateAligment()) );
}

void
DefaultEditor::updateActions()
{
  updateCharFmt();
  updateAligment();
  updateFont();
}

void
DefaultEditor::updateCharFmt()
{
  m_actionBold->setChecked( m_textEdit->bold() );
  m_actionItalic->setChecked( m_textEdit->italic() );
  m_actionUnderline->setChecked( m_textEdit->underline() );
}

void
DefaultEditor::updateAligment()
{
  int align = m_textEdit->alignment();

  switch ( align ) {
  case AlignRight:
    m_actionAlignRight->setChecked( true );
    break;
  case AlignCenter:
    m_actionAlignCenter->setChecked( true );
    break;
  case AlignLeft:
    m_actionAlignLeft->setChecked( true );
    break;
  case AlignJustify:
    m_actionAlignJustify->setChecked( true );
    break;
  default:
    break;
  }
}

void
DefaultEditor::updateFont()
{
  if ( m_textEdit->pointSize() > 0 )
    m_actionFontSize->setFontSize( m_textEdit->pointSize() );
  m_actionFont->setFont( m_textEdit->family() );
}

void
DefaultEditor::formatColor()
{
  TQColor col;

  int s = KColorDialog::getColor( col, m_textEdit->color(), m_textEdit );
  if ( s != TQDialog::Accepted )
    return;

  m_textEdit->setColor( col );
}

void
DefaultEditor::setAlignLeft( bool yes )
{
  if ( yes )
    m_textEdit->setAlignment( AlignLeft );
}

void
DefaultEditor::setAlignRight( bool yes )
{
  if ( yes )
    m_textEdit->setAlignment( AlignRight );
}

void
DefaultEditor::setAlignCenter( bool yes )
{
  if ( yes )
    m_textEdit->setAlignment( AlignCenter );
}

void
DefaultEditor::setAlignJustify( bool yes )
{
  if ( yes )
    m_textEdit->setAlignment( AlignJustify );
}

//
// Content Actions
//

bool
DefaultEditor::open()
{
  KURL url = KFileDialog::getOpenURL();
  if ( url.isEmpty() )
    return false;

  //fixme
  //return openURL( url );
  return true;
}

bool
DefaultEditor::saveAs()
{
  KURL url = KFileDialog::getSaveURL();
  if ( url.isEmpty() )
    return false;

  //FIXME
  //return KParts::ReadWritePart::saveAs( url );
  return true;
}

void
DefaultEditor::checkSpelling()
{
  TQString s;
  if ( m_textEdit->hasSelectedText() )
    s = m_textEdit->selectedText();
  else
    s = m_textEdit->text();

  //KSpell::modalCheck( s );
}

bool
DefaultEditor::print()
{
  return true;
}

#include "defaulteditor.moc"
